/**********************************************************************
 *<
	FILE: area.cpp

	DESCRIPTION:  surface area of the object

	CREATED BY: Andrey Zmievski

	HISTORY: started - March 1, 1997

 *>	Copyright (c) 1997, All Rights Reserved.
 **********************************************************************/
#include "area.h"
#include /* */"utilapi.h"
#include /* */"stdmat.h"
#include <shlobj.h>

#define AREA_CLASS_ID Class_ID(0x35bd658d, 0x5773726f)

class Area : public UtilityObj {
	public:
		IUtil *iu;
		Interface *ip;
		HWND hPanel;
		ICustButton *iPick;
		float area;

		Area();			// Constructor
		void BeginEditParams(Interface *ip,IUtil *iu);
		void EndEditParams(Interface *ip,IUtil *iu);
		void DeleteThis() {}

		void Init(HWND hWnd);	 // Initialization
		void Destroy(HWND hWnd);

		void GetArea(INode *node);	// Grunt of work is done here
	};
static Area theArea;

class AreaClassDesc:public ClassDesc {
	public:
	int 			IsPublic() {return 1;}
	void *			Create(BOOL loading = FALSE) {return &theArea;}
	const TCHAR *	ClassName() {return GetString(IDS_RB_AREA);}
	SClass_ID		SuperClassID() {return UTILITY_CLASS_ID;}
	Class_ID		ClassID() {return AREA_CLASS_ID;}
	const TCHAR* 	Category() {return _T("Terralux");}
	};

static AreaClassDesc AreaDesc;
ClassDesc* GetAreaDesc() {return &AreaDesc;}

class AreaPickNodeCallback : public PickNodeCallback {
	public:         
		BOOL Filter(INode *node);	// returns TRUE if the node is the one we can use
};

static AreaPickNodeCallback thePickFilt;

class AreaPickModeCallback : public PickModeCallback {
	public:         
		BOOL HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags);
		BOOL Pick(IObjParam *ip,ViewExp *vpt);
                          
        void EnterMode(IObjParam *ip) {theArea.iPick->SetCheck(TRUE);}
        void ExitMode(IObjParam *ip) {theArea.iPick->SetCheck(FALSE);}

        BOOL RightClick(IObjParam *ip,ViewExp *vpt) {return TRUE;}
};


static AreaPickModeCallback thePickMode;

class NullView : public View {
	public:
	Point2 ViewToScreen(Point3 p)
	{ return Point2(p.x,p.y); }

	NullView() {	
		worldToView.IdentityMatrix();
		screenW=640.0f; screenH = 480.0f;
	}
};

//-----------------------------------------------------

// Returns true if something was hit
BOOL AreaPickModeCallback::HitTest(
	IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags)
    {
    return ip->PickNode(hWnd,m,&thePickFilt)?TRUE:FALSE;
}

// This is called when a user picks something
BOOL AreaPickModeCallback::Pick(IObjParam *ip,ViewExp *vpt)
    {
    INode *node = vpt->GetClosestHit();
    if (node) {
	    theArea.GetArea(node);
    }
    return TRUE;
}

// Filters the incoming nodes
BOOL AreaPickNodeCallback::Filter(INode *node) {
	ObjectState os = node->EvalWorldState(theArea.ip->GetTime());
    if (os.obj->SuperClassID()==GEOMOBJECT_CLASS_ID &&
		os.obj->IsRenderable()) return TRUE;  // only if actual geometry exists
    else return FALSE;
}

static BOOL CALLBACK AreaDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{
	switch (msg) {
		case WM_INITDIALOG:
			theArea.Init(hWnd);
			break;
		
		case WM_DESTROY:
			theArea.Destroy(hWnd);
			break;

		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK:
					theArea.iu->CloseUtility();
					break;				
		
				case IDC_AREA_PICK:
					theArea.ip->SetPickMode(&thePickMode);
					break;
				}
			break;

		default:
			return FALSE;
		}
	return TRUE; 
	}

Area::Area()
	{
	iu = NULL;
	ip = NULL;	
	hPanel = NULL;	
	iPick = NULL;
	}

void Area::BeginEditParams(Interface *ip,IUtil *iu) 
	{
	this->iu = iu;
	this->ip = ip;
	hPanel = ip->AddRollupPage(
		hInstance,
		MAKEINTRESOURCE(IDD_AREA_PANEL),
		AreaDlgProc,
		GetString(IDS_RB_AREA),
		0);
	}
	
void Area::EndEditParams(Interface *ip,IUtil *iu) 
	{
	ip->ClearPickMode();
	this->iu = NULL;
	this->ip = NULL;
	ip->DeleteRollupPage(hPanel);
	hPanel = NULL;
}

void Area::Init(HWND hWnd)
	{
	iPick = GetICustButton(GetDlgItem(hWnd,IDC_AREA_PICK));
	iPick->SetType(CBT_CHECK);			// Set the button to check-type instead of push
	iPick->SetHighlightColor(GREEN_WASH);	// And highlight it in green
	}

void Area::Destroy(HWND hWnd)
	{
	ReleaseICustButton(iPick);
	iPick = NULL;
}

void Area::GetArea(INode *node) {
    // Get the object that is referenced at the time	
	ObjectState os = node->EvalWorldState(theArea.ip->GetTime());
	// Just to make sure it's a geometric object before we typecast
	assert(os.obj->SuperClassID()==GEOMOBJECT_CLASS_ID);
	BOOL needDel;
	NullView nullView; // Set up a null viewport so that we get the right mesh..
	Mesh *mesh = (((GeomObject*)os.obj)->GetRenderMesh(ip->GetTime(),node,nullView,needDel)); // Type cast to a geometric object and then retrieve it's rendermesh
	if (!mesh) return; // Make sure there is a mesh to work with

	Matrix3 tm = node->GetObjTMAfterWSM(theArea.ip->GetTime()); // Get the transformation 
	
	area = 0.0f;		// reset the total area
	for (int i=0; i<mesh->getNumFaces(); i++) {
		Face f = mesh->faces[i];		// get 1 face

		Point3 e1 = mesh->verts[f.v[1]]-mesh->verts[f.v[0]];		// get first edge
		Point3 e2 = mesh->verts[f.v[2]]-mesh->verts[f.v[0]];		// get second edge
		e1 = VectorTransform(tm, e1);		// transform with the matrix
		e2 = VectorTransform(tm, e2);
		Point3 res = CrossProd(e1, e2);	// get crossproduct
		area += Length(res)/2.0f;		// and divide its length by 2 to get triangle area
	}
	char str[256];
	sprintf(str, "The area of '%s' is %.2f", node->GetName(), area);
	MessageBox(ip->GetMAXHWnd(), str, "Surface Area", MB_OK | MB_ICONINFORMATION);

	if (needDel) delete mesh;
}
